// firebase-config.js se config yahan load hogi
firebase.initializeApp(firebaseConfig);
const db = firebase.firestore();
const roomRef = db.collection('calls');

const servers = {
    iceServers: [{ urls: ['stun:stun1.l.google.com:19302', 'stun:stun2.l.google.com:19302'] }]
};

let localStream = null;
let peerConnections = {}; // Multiple users ko track karne ke liye

async function init() {
    // 1. Mic permission lena
    localStream = await navigator.mediaDevices.getUserMedia({ audio: true, video: false });
    
    // Default: Mic mute rakhenge
    localStream.getAudioTracks()[0].enabled = false;

    // 2. Presence setup: Firestore mein batana ki main online hoon
    const myStatus = db.collection('presence').doc(USERNAME);
    await myStatus.set({
        username: USERNAME,
        isSpeaking: false,
        joinedAt: Date.now()
    });

    // Jab hum exit karein toh presence delete ho jaye
    window.addEventListener('beforeunload', () => myStatus.delete());

    // 3. Listen for other users joining
    db.collection('presence').onSnapshot(snapshot => {
        updateUI(snapshot);
        snapshot.docChanges().forEach(async (change) => {
            if (change.type === 'added' && change.doc.id !== USERNAME) {
                // Naya banda aaya! Uske liye connection banao
                createPeerConnection(change.doc.id);
            }
        });
    });

    setupPTT();
}

async function createPeerConnection(remoteUser) {
    const pc = new RTCPeerConnection(servers);
    peerConnections[remoteUser] = pc;

    // Local audio track add karo
    localStream.getTracks().forEach(track => pc.addTrack(track, localStream));

    // Jab remote audio aaye
    pc.ontrack = (event) => {
        const remoteAudio = new Audio();
        remoteAudio.srcObject = event.streams[0];
        remoteAudio.autoplay = true;
    };

    // ICE Candidates exchange
    pc.onicecandidate = (event) => {
        if (event.candidate) {
            db.collection('candidates').add({
                to: remoteUser,
                from: USERNAME,
                candidate: event.candidate.toJSON()
            });
        }
    };

    // Create Offer
    const offer = await pc.createOffer();
    await pc.setLocalDescription(offer);
    
    db.collection('offers').doc(remoteUser).set({
        from: USERNAME,
        offer: { type: offer.type, sdp: offer.sdp }
    });
}

// Logic for Press-To-Talk (PTT)
function setupPTT() {
    const micBtn = document.getElementById('micBtn');
    if (!micBtn) return;

    const startMic = () => {
        localStream.getAudioTracks()[0].enabled = true;
        db.collection('presence').doc(USERNAME).update({ isSpeaking: true });
        micBtn.classList.add('speaking');
    };

    const stopMic = () => {
        localStream.getAudioTracks()[0].enabled = false;
        db.collection('presence').doc(USERNAME).update({ isSpeaking: false });
        micBtn.classList.remove('speaking');
    };

    // Desktop
    micBtn.addEventListener('mousedown', startMic);
    window.addEventListener('mouseup', stopMic);

    // Mobile
    micBtn.addEventListener('touchstart', (e) => { e.preventDefault(); startMic(); });
    micBtn.addEventListener('touchend', stopMic);
}

function updateUI(snapshot) {
    const list = document.getElementById('user-list');
    list.innerHTML = '';
    snapshot.forEach(doc => {
        const data = doc.data();
        const div = document.createElement('div');
        div.className = `user-row ${data.isSpeaking ? 'speaking' : ''}`;
        div.innerHTML = `
            <span>${data.username}</span>
            <span class="status-dot"></span>
        `;
        list.appendChild(div);
    });
}

init();